<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\RosterAssign;
use App\Models\EmployeeAttendance;
use App\Models\LeaveRequest;
use App\Models\EmpCompanyDetails;
use App\Models\Sites;
use App\Models\Project;
use Illuminate\Support\Collection;
use App\Models\EmpTeam;
use App\Models\EmpPersonalDetails;
use Carbon\Carbon;
use App\Models\EmpDocuments;
use App\Models\RequiredDocument;
use DB;
use App\Models\RiseSalary;
use App\Models\Salary;
use Illuminate\Database\Eloquent\Builder;

class DashboardController extends Controller
{

    public $current_date;

    public function __construct()
    {
        $this->current_date = date('Y-m-d');
    }

    public function index(Request $request)
    {
        $data['total_employees']  =   EmpCompanyDetails::where('del',0)->where('compeleted','1')->count();


        $data['total_sites']      =   Sites::where('del',0)->count();


        $employees_ids = RosterAssign::join('emp_company_details','roster_assigns.assign_to','emp_company_details.id')
                                    ->where('emp_company_details.del',0)
                                    ->where('emp_company_details.status',1)
                                    ->where('roster_assigns.schedule_date', $this->current_date)
                                    ->groupBy('roster_assigns.assign_to')
                                    ->pluck('roster_assigns.assign_to')
                                    ->toArray();

        $data['total_roster_employees'] = count($employees_ids);

        $present_employee_ids = EmployeeAttendance::where('date',$this->current_date)
                                    ->wherein('employee_id',$employees_ids)
                                    ->groupBy('employee_id')
                                    ->pluck('employee_id')
                                    ->toArray();

        $data['total_present'] = count($present_employee_ids);

        $leave_employee_ids = LeaveRequest::where('from',">=",$this->current_date)
                                            ->where('to',"<=",$this->current_date)
                                            ->where('status',1)
                                            ->pluck('employee_id')
                                            ->toArray();

                                
        $data['total_leaves'] = count($leave_employee_ids);

        $data['total_absent'] = $data['total_roster_employees'] - $data['total_present'] - $data['total_leaves'];

        $data['today_present'] = EmployeeAttendance::where('date',$this->current_date)
                                    ->with('present_employee_details','employee_site')
                                    ->get();

        // Calculate the number of employees at each site based on site name
        $employeeCountBySiteName = $data['today_present']->groupBy(function ($attendance) {
            return $attendance->employee_site->title ?? 'Global Site'; // Group by site name
        })->map(function ($site) {
            return [
                'employee_count' => $site->count(),  // Count of employees
                'check_in_count' => $site->whereNotNull('check_in')->count(),  // Count of check-ins
                'check_out_count' => $site->whereNotNull('check_out')->count()  // Count of check-outs
            ];
        });

        // Convert to a simple array (if needed)
        $data['sites_employee'] = $employeeCountBySiteName->toArray();

        $data['today_leaves'] = LeaveRequest::with('leave_employee_details')
                                    ->where('from',">=",$this->current_date)
                                    ->where('to',"<=",$this->current_date)
                                    ->get();
        
        ////////////// TEAMS ////////////////////
        $emp_teams = EmpTeam::with(['emp_teams_member' => function ($query) {
            $query->whereHas('roster_assigns');
        }])->where('del', 0)->get();
        
        $teams_response = [];
        
        foreach ($emp_teams as $team) {
            $total_employees_with_roster = $team->emp_teams_member->count();
            $total_employees_with_attendance = 0;
            $total_employees_without_attendance = 0;
        
            // Loop through each team member
            foreach ($team->emp_teams_member as $team_member) {
                // Check if the employee has attendance for the day
                if ($team_member->roster_assigns && $team_member->roster_assigns->team_employee_attendance) {
                    $total_employees_with_attendance++;
                } else {
                    $total_employees_without_attendance++;
                }
            }
        
            // Add the data for this team to the response array
            $teams_response[] = [
                'team_id' => $team->id,
                'team_title' => $team->title,
                'total_employees_with_roster' => $total_employees_with_roster,
                'total_employees_with_attendance' => $total_employees_with_attendance,
                'total_employees_without_attendance' => $total_employees_without_attendance,
            ];
        }
        $data['teams'] = $teams_response;
        ////////////// TEAMS ////////////////////


        $nextMonth = Carbon::now()->addMonth()->format('Y-m');

        $riseSalaries = EmpPersonalDetails::join('emp_company_details', 'emp_personal_details.emp_id', '=', 'emp_company_details.id')
            ->select(
                'emp_personal_details.first_name',
                'emp_personal_details.middle_name',
                'emp_personal_details.last_name',
                'emp_personal_details.emp_id'
            )
            ->addSelect([
                'last_salary' => Salary::select('basic_salary')
                    ->whereColumn('employee_id', 'emp_personal_details.emp_id')
                    ->where('to', $nextMonth)
                    ->orderByDesc('created_at')
                    ->limit(1)
            ])
            ->where('emp_personal_details.del', 0)
            ->where('emp_company_details.del', 0)
            // Only show if there is a salary for next month
            ->whereExists(function($query) use ($nextMonth) {
                $query->select(DB::raw(1))
                    ->from('salaries') // Make sure this matches your actual table name
                    ->whereColumn('employee_id', 'emp_personal_details.emp_id')
                    ->where('to', $nextMonth);
            })
            ->distinct()
            ->get();
        
        $data['rise_salaries'] = $riseSalaries;

        return view('dashboard', compact('data'));
    }

    public function expiringDocuments(Request $request) {
        $today = Carbon::today();
        $near_expiry_date = $today->addDays(30); // Documents expiring in the next 30 days
    
        // Base query for fetching documents
        $query = DB::table('emp_documents as expiry_doc')
            ->join('emp_personal_details', 'emp_personal_details.emp_id', '=', 'expiry_doc.emp_id')
            ->join('emp_company_details', 'emp_personal_details.emp_id', '=', 'emp_company_details.id')
            ->join('emp_teams_members', 'emp_personal_details.emp_id', '=', 'emp_teams_members.emp_id')
            ->join('emp_teams', 'emp_teams_members.team_id', '=', 'emp_teams.id')
            ->join('required_documents', 'required_documents.id', '=', 'expiry_doc.required_document')
            ->join('required_document_fields as expiry_field', 'expiry_field.id', '=', 'expiry_doc.required_document_field')
            ->leftJoin('emp_documents as file_doc', function($join) {
                $join->on('expiry_doc.emp_id', '=', 'file_doc.emp_id')
                     ->on('expiry_doc.required_document', '=', 'file_doc.required_document');
            })
            ->join('required_document_fields as file_field', 'file_field.id', '=', 'file_doc.required_document_field')
            ->where('expiry_field.field_name', 'Expiry Date')  // Ensure we're filtering for expiry date field
            ->where('file_field.field_type', 'file')  // Ensure we're filtering for file fields
            ->whereDate('expiry_doc.value', '<=', $near_expiry_date)  // Filter based on expiry date
            ->where('expiry_doc.del', '0')  // Filter non-deleted expiry records
            ->where('file_doc.del', '0')    // Filter non-deleted file records
            ->where('emp_company_details.del', '0')
            ->where('emp_company_details.compeleted', '1')
            ->select(
                'emp_personal_details.emp_id',
                'emp_personal_details.first_name',
                'emp_personal_details.middle_name',
                'emp_personal_details.last_name',
                DB::raw("GROUP_CONCAT(emp_teams.title SEPARATOR ', ') as team_titles"),  // Combine multiple team titles
                'required_documents.id as document_id',
                'required_documents.title as document_title',
                'expiry_doc.value as expiry_date',  // Expiry date
                'file_doc.value as document_path',   // File path from the file_doc
                DB::raw("DATEDIFF(expiry_doc.value, CURDATE()) as remaining_days")  // Calculate remaining days
            )
            ->groupBy(
                'emp_personal_details.emp_id',
                'emp_personal_details.first_name',
                'emp_personal_details.middle_name',
                'emp_personal_details.last_name',
                'required_documents.id',
                'required_documents.title',
                'expiry_doc.value',
                'file_doc.value'
            );
    
        $count = DB::table(DB::raw("({$query->toSql()}) as sub"))->mergeBindings($query)->count();
    
        // Apply pagination
        $start = $request->input('from', 0);
        $limit = 10;  // Set the limit for pagination
        $query->offset($start)->limit($limit);
    
        // Get the paginated results
        $query_result = $query->get();
    
        $data =  [
            'expiring_documents' => $query_result,
        ];
    
        return response()->json([
            'message' => 'Get employees with expiring documents successfully.',
            'data' => $data,
            'count' => $count,
        ], 200);
    }
    
    public function missingDocuments(Request $request)
    {
        $limit = 10;  
        $start = $request->input('from', 0);

        // Subquery to get the uploaded documents for each employee
        $uploadedDocuments = DB::table('emp_documents')
            ->select('emp_id', DB::raw('GROUP_CONCAT(required_document) as uploaded_docs'))
            ->groupBy('emp_id');

        $query = DB::table('emp_personal_details')
            ->join('emp_teams_members', 'emp_personal_details.emp_id', '=', 'emp_teams_members.emp_id')
            ->join('emp_teams', 'emp_teams_members.team_id', '=', 'emp_teams.id')
            ->join('emp_company_details', 'emp_personal_details.emp_id', '=', 'emp_company_details.id') // Join employee company details
            ->crossJoin('required_documents')
            ->join('required_document_fields', 'required_documents.id', '=', 'required_document_fields.doc_id')
            ->leftJoinSub($uploadedDocuments, 'uploaded_docs', function ($join) {
                $join->on('emp_personal_details.emp_id', '=', 'uploaded_docs.emp_id');
            })
            ->whereRaw('FIND_IN_SET(required_documents.id, IFNULL(uploaded_docs.uploaded_docs, "")) = 0')
            ->where('emp_personal_details.del', '0')
            ->where('required_documents.del', '0')
            ->where('required_document_fields.field_required', 1)
            ->where(function ($query) {
                $query->where('required_documents.for_who', '2')
                    ->orWhere(function ($query) {
                        $query->whereColumn('required_documents.for_who', 'emp_company_details.user_type');
                    });
            })
            ->select(
                'emp_personal_details.emp_id',
                'emp_personal_details.first_name',
                'emp_personal_details.middle_name',
                'emp_personal_details.last_name',
                DB::raw("GROUP_CONCAT(DISTINCT emp_teams.title SEPARATOR ', ') as team_titles"),
                'required_documents.title as document_title',
                'emp_company_details.user_type'
            )
            ->groupBy(
                'emp_personal_details.emp_id',
                'emp_personal_details.first_name',
                'emp_personal_details.middle_name',
                'emp_personal_details.last_name',
                'required_documents.title',
                'emp_company_details.user_type',
            );

        $count = DB::table(DB::raw("({$query->toSql()}) as sub"))
            ->mergeBindings($query)
            ->count();

        $results = $query->offset($start)->limit($limit)->get();

        return response()->json([
            'message' => 'Get employees with missing documents successfully.',
            'data' => ['missing_documents' => $results],
            'count' => $count,
        ], 200);
    }

    public function absentEmployees(Request $request){

        $employees_ids = RosterAssign::join('emp_company_details','roster_assigns.assign_to','emp_company_details.id')
                            ->where('emp_company_details.del',0)
                            ->where('emp_company_details.status',1)
                            ->where('roster_assigns.schedule_date', $this->current_date)
                            ->groupBy('roster_assigns.assign_to')
                            ->pluck('roster_assigns.assign_to')
                            ->toArray();
        $present_employee_ids = EmployeeAttendance::where('date',$this->current_date)
                                    ->wherein('employee_id',$employees_ids)
                                    ->groupBy('employee_id')
                                    ->pluck('employee_id')
                                    ->toArray();
        $leave_employee_ids = LeaveRequest::where('from',">=",$this->current_date)
                                ->where('to',"<=",$this->current_date)
                                ->where('status',1)
                                ->pluck('employee_id')
                                ->toArray();

        $absent_employee_id = array_values(array_diff($employees_ids,array_merge($present_employee_ids,$leave_employee_ids)));

        $absent_employees = EmpCompanyDetails::join('emp_personal_details','emp_company_details.id','emp_personal_details.emp_id')
                                    ->join('roles','emp_company_details.access_role','roles.code')
                                    ->wherein('emp_company_details.id', $absent_employee_id)
                                    ->select('emp_company_details.employee_email','emp_personal_details.*','roles.title');

        $count = $absent_employees->count();
        $start = $request->input('from', 0);
        $limit = 10;  // Set the limit for pagination
        $absent_employees->offset($start)->limit($limit);
    
        // Get the paginated results
        $query_result = $absent_employees->get();
    
        $data =  [
            'absent_employees' => $query_result,
        ];

        return response()->json([
            'message' => 'Get absent employees successfully.',
            'data' => $data,
            'count' => $count,
        ], 200);
    }

    public function presentEmployees(Request $request){
        // dd("Heloo buddy");
        $present_employees = EmployeeAttendance::where('date',$this->current_date)
                                    ->with('present_employee_details','employee_site');

        $count = $present_employees->count();
        $start = $request->input('from', 0);
        $limit = 10;  // Set the limit for pagination
        $present_employees->offset($start)->limit($limit);
    
        // Get the paginated results
        $query_result = $present_employees->get();
        
        foreach ($query_result as &$present) {
            $present['formatted_date'] = formatDate($present['date']);
            $present['formatted_check_in'] = formatTime($present['check_in']);
            $present['formatted_check_out'] = formatTime($present['check_out']);
        }
        
        $data =  [
            'today_present' => $query_result,
        ];

        return response()->json([
            'message' => 'Get employees with expiring documents successfully.',
            'data' => $data,
            'count' => $count,
        ], 200);
    }
}
