<?php

namespace App\Http\Controllers;

use PDF;
use App\Models\Fine;
use App\Models\Salary;
use App\Models\Payroll;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Models\EmpCompanyDetails;
use App\Models\EmployeeAttendance;
use App\Models\EmpPersonalDetails;
use App\Models\LeavePackage;
use App\Models\Overtime;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use App\Models\RosterAssign;
use App\Models\RosterTemplate;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Traits\EmailTrait;
use App\Jobs\GenerateAllPayrollReceiptsJob;
use App\Models\PublicHoliday;
use Carbon\CarbonPeriod;

class PayRollController extends Controller
{
    use EmailTrait;
    public function generateReceiptsForAllEmployees(Request $request)
    {
        GenerateAllPayrollReceiptsJob::dispatch();

        return response()->json([
            'success' => true,
            'message' => 'Payroll receipts generation has been queued and will be processed in the background.',
        ]);
    }
    public function index(Request $request)
    {
        $filters = $request->only(['employee', 'pay_year', 'pay_month', 'status']);
        $payrolls = Payroll::with('employee')
            ->when($filters['employee'] ?? null, function ($query, $employeeId) {
                $query->where('employee_id', $employeeId);
            })
            ->when($filters['pay_year'] ?? null, function ($query, $year) {
                $query->where('pay_year', $year);
            })
            ->when($filters['pay_month'] ?? null, function ($query, $month) {
                $query->where('pay_month', $month);
            })
            ->when($filters['status'] ?? null, function ($query, $status) {
                $query->where('status', $status);
            })
            ->get()
            ->map(function ($payroll) {
                // Calculate total fines for this payroll
                $totalFines = Fine::where('employee_id', $payroll->employee_id)
                    ->whereYear('date', $payroll->pay_year)
                    ->whereMonth('date', $payroll->pay_month)
                    ->where('type', 0) // Only fines, not bonuses
                    ->sum('fine_amount');
                
                $payroll->total_fines = $totalFines;
                return $payroll;
            });

        // Convert back to pagination
        $perPage = 10;
        $currentPage = request()->get('page', 1);
        $payrolls = new \Illuminate\Pagination\LengthAwarePaginator(
            $payrolls->forPage($currentPage, $perPage),
            $payrolls->count(),
            $perPage,
            $currentPage,
            ['path' => request()->url(), 'query' => request()->query()]
        );

        $get_employees = EmpCompanyDetails::with('EmpPersonalDetails')
            ->where('del', 0)
            ->where('compeleted', 1)
            ->where('status', 1)
            ->get();

           
        return view('Payroll.index', compact('payrolls', 'filters', 'get_employees'));
    }

    public function fine_management(Request $request)
    {
        // Only accept the filters that you now want to use.
        $filters = $request->only(['employee', 'fine_amount', 'type', 'date', 'from_date', 'to_date']);
        
        // Handle date range picker format (DD-MM-YYYY / DD-MM-YYYY)
        if ($request->filled('date')) {
            $dateRange = explode(' / ', $request->date);
            if (count($dateRange) == 2) {
                $filters['from_date'] = \Carbon\Carbon::createFromFormat('d-m-Y', trim($dateRange[0]))->format('Y-m-d');
                $filters['to_date'] = \Carbon\Carbon::createFromFormat('d-m-Y', trim($dateRange[1]))->format('Y-m-d');
            }
        }
    
        $fines = Fine::with('employee')
            ->when($filters['employee'] ?? null, function ($query, $employeeId) {
                $query->where('employee_id', $employeeId);
            })
            ->when($filters['fine_amount'] ?? null, function ($query, $amount) {
                $query->where('fine_amount', $amount);
            })
            ->when(isset($filters['type']) && $filters['type'] !== '', function ($query) use ($filters) {
                $query->where('type', $filters['type']);
            })
            ->when($filters['from_date'] ?? null, function ($query, $fromDate) {
                $query->whereDate('date', '>=', $fromDate);
            })
            ->when($filters['to_date'] ?? null, function ($query, $toDate) {
                $query->whereDate('date', '<=', $toDate);
            })
            ->latest('created_at')
            ->paginate(10);
    
        $get_employees = EmpCompanyDetails::with('EmpPersonalDetails')
            ->where('del', 0)
            ->where('compeleted', 1)
            ->where('status', 1)
            ->get();
    
        return view('Fine.index', compact('fines', 'filters', 'get_employees'));
    }
    

    public function salary_index(Request $request)
    {
        $filters = $request->only(['employee', 'from', 'to', 'working_hours', 'basic_salary']);
        $fines = Salary::with('employee')
            ->when($filters['employee'] ?? null, function ($query, $employeeId) {
                $query->where('employee_id', $employeeId);
            })
            ->when($filters['from'] ?? null, function ($query, $year) {
                $query->where('from', $year);
            })
            ->when($filters['to'] ?? null, function ($query, $month) {
                $query->where('to', $month);
            })
            ->when($filters['working_hours'] ?? null, function ($query, $status) {
                $query->where('working_hours', $status);
            })
            ->when($filters['basic_salary'] ?? null, function ($query, $status) {
                $query->where('basic_salary', $status);
            })
            ->paginate(10);

        $get_employees = EmpCompanyDetails::with('EmpPersonalDetails')
            ->where('del', 0)
            ->where('compeleted', 1)
            ->where('status', 1)
            ->get();

        return view('Salary.index', compact('fines', 'filters', 'get_employees'));
    }

    public function create(Request $request)
    {
        $get_employes = EmpCompanyDetails::with('EmpPersonalDetails')->where('del', 0)->where('compeleted', 1)->where('status', 1)->get();
        $get_companies = [];
        return view('Payroll.create', compact('get_employes', 'get_companies'));
    }

    public function salary_create(Request $request)
    {
        $get_employes = EmpCompanyDetails::with('EmpPersonalDetails')->where('del', 0)->where('compeleted', 1)->where('status', 1)->get();
        $get_companies = [];
        return view('Salary.create', compact('get_employes', 'get_companies'));
    }

    public function fine_create(Request $request)
    {
        $get_employes = EmpCompanyDetails::with('EmpPersonalDetails')->where('del', 0)->where('compeleted', 1)->where('status', 1)->get();
        $get_companies = [];
        return view('Fine.create', compact('get_employes', 'get_companies'));
    }

    public function store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'employee_id' => 'required|integer',
                'pay_year' => 'required|integer',
                'pay_month' => 'required|integer',
                'basic_salary' => 'required|numeric|min:0',
                'working_hours' => 'required|numeric|min:0',
                'hours_spended' => 'required|numeric|min:0',
                'calculated_salary' => 'required|numeric|min:0',
            ]);

            $payroll = Payroll::create([
                'employee_id' => $validatedData['employee_id'],
                'pay_year' => $validatedData['pay_year'],
                'pay_month' => $validatedData['pay_month'],
                'basic_salary' => $validatedData['basic_salary'],
                'working_hours' => $validatedData['working_hours'],
                'hours_spent' => $validatedData['hours_spended'],
                'calculated_salary' => $validatedData['calculated_salary'],
                'status' => 0,
            ]);

            return redirect()->route('payroll.index')->with('success', 'Payroll changed successfully!');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'errors' => $e->errors(),
            ], 422);
        }
    }

    public function fine_store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'employee_id' => 'required|integer',
                'fine_date' => 'required',
                'fine_amount' => 'required|numeric|min:0',
                'fine_reason' => 'required|string|max:250',
                'type' => 'required|numeric|min:0',
            ]);

            $payroll = Fine::create([
                'employee_id' => $validatedData['employee_id'],
                'date' => $validatedData['fine_date'],
                'fine_amount' => $validatedData['fine_amount'],
                'type' => $validatedData['type'],
                'fine_reason' => $validatedData['fine_reason'],
                'is_attendance_related' => 0,
            ]);

            return redirect()->route('fine.management')->with('success', 'Fine charged successfully!');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'errors' => $e->errors(),
            ], 422);
        }
    }

    public function salary_store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'employee_id' => 'required|integer',
                'from' => 'required|date_format:Y-m',
                'to' => 'required|date_format:Y-m',
                'basic_salary' => 'required|numeric|min:0',
                'working_hours' => 'required|string',
            ]);



            $payroll = Salary::create([
                'employee_id' => $validatedData['employee_id'],
                'from' => $validatedData['from'],
                'to' => $validatedData['to'],
                'basic_salary' => $validatedData['basic_salary'],
                'working_hours' => $validatedData['working_hours'],
            ]);

            return redirect()->route('salary.index')->with('success', 'Fine charged successfully!');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'errors' => $e->errors(),
            ], 422);
        }
    }

    public function edit($id)
    {
        $payroll = Payroll::findOrFail($id);
        $get_employees = EmpCompanyDetails::with('EmpPersonalDetails')->where('del', 0)->where('compeleted', 1)->where('status', 1)->get();

        return view('Payroll.edit', compact('payroll', 'get_employees'));
    }

    public function fine_edit($id)
    {
        $fine = Fine::findOrFail($id);
        $get_employees = EmpCompanyDetails::with('EmpPersonalDetails')->where('del', 0)->where('compeleted', 1)->where('status', 1)->get();

        return view('Fine.edit', compact('fine', 'get_employees'));
    }

    public function salary_edit($id)
    {
        $salary = Salary::findOrFail($id);
        $get_employees = EmpCompanyDetails::with('EmpPersonalDetails')->where('del', 0)->where('compeleted', 1)->where('status', 1)->get();

        return view('Salary.edit', compact('salary', 'get_employees'));
    }

    public function update(Request $request, $id)
    {
        $validatedData = $request->validate([
            'employee_id' => 'required|integer',
            'pay_year' => 'required|integer',
            'pay_month' => 'required|integer',
            'basic_salary' => 'required|numeric|min:0',
            'working_hours' => 'required|numeric|min:0',
            'hours_spended' => 'required',
            'calculated_salary' => 'required|numeric|min:0',
        ]);

        // Parse hours_spended (HH:MM or decimal)
        $hours_spend = 0;
        if (strpos($request->hours_spended, ':') !== false) {
            // Parse HH:MM format
            [$hours, $minutes] = explode(':', $request->hours_spended);
            $hours_spend = ((int)$hours * 60) + (int)$minutes;
        } else {
            // Fallback: treat as decimal hours
            $hours_spend = (float)$request->hours_spended * 60;
        }
        $working_hours = (float)$request->working_hours * 60;
        $payroll = Payroll::findOrFail($id);
        $payroll->update([
            'employee_id' => $validatedData['employee_id'],
            'pay_year' => $validatedData['pay_year'],
            'pay_month' => $validatedData['pay_month'],
            'basic_salary' => $validatedData['basic_salary'],
            'working_hours' => $working_hours,
            'hours_spent' => $hours_spend,
            'calculated_salary' => $validatedData['calculated_salary'],
            'status' => 0,
        ]);

        return redirect()->route('payroll.index')->with('success', 'Payroll updated successfully!');
    }

    public function fine_update(Request $request, $id)
    {
        $validatedData = $request->validate([
            'employee_id' => 'required|integer',
            'fine_date' => 'required',
            'fine_amount' => 'required|numeric|min:0',
            'fine_reason' => 'required|string|max:250',
            'type' => 'required|numeric',
        ]);
    
        $fine = Fine::findOrFail($id);
        $fine->update([
            'employee_id' => $validatedData['employee_id'],
            'date' => $validatedData['fine_date'],
            'type' => $validatedData['type'],
            'fine_amount' => $validatedData['fine_amount'],
            'fine_reason' => $validatedData['fine_reason'],
        ]);

        return redirect()->route('fine.management')->with('success', 'Fine updated successfully!');
    }

    public function salary_update(Request $request, $id)
    {
        $validatedData = $request->validate([
            'employee_id' => 'required|integer',
            'from' => 'required|date_format:Y-m',
            'to' => 'required|date_format:Y-m',
            'basic_salary' => 'required|numeric|min:0',
            'working_hours' => 'required|string',
        ]);

        $salary = Salary::findOrFail($id);
        $salary->update([
            'employee_id' => $validatedData['employee_id'],
            'from' => $validatedData['from'],
            'to' => $validatedData['to'],
            'basic_salary' => $validatedData['basic_salary'],
            'working_hours' => $validatedData['working_hours'],
        ]);

        return redirect()->route('salary.index')->with('success', 'salary updated successfully!');
    }

    public function toggleStatus($id)
    {
        $payroll = Payroll::findOrFail($id);
        $payroll->status = $payroll->status ? 0 : 1;
        $payroll->save();

        return back()->with('success', 'Status changed successfully!');
    }

    public function destroy($id)
    {
        $payroll = Payroll::findOrFail($id);
        
        // Fine::where('employee_id', $payroll->employee_id)
        //     ->whereYear('date', $payroll->pay_year)
        //     ->whereMonth('date', $payroll->pay_month)
        //     ->delete();
        
        $payroll->delete();

        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Payroll and related fines deleted successfully.'
            ]);
        }

        return redirect()->route('payroll.index')->with('success', 'Payroll and related fines deleted successfully.');
    }

    public function fine_destroy($id)
    {
        $fine = Fine::findOrFail($id);

        $fine->delete();

        return redirect()->back()->with('success', 'Fine deleted successfully.');
    }

    public function getEmployeeFines(Request $request)
    {
        // Validate the incoming request
        $validated = $request->validate([
            'from' => 'required|date',
            'to'   => 'required|date|after_or_equal:from_date',
        ]);

        // Get the authenticated employee's ID
        $employeeId = auth()->user()->id;

        // Query the employee_fines table to fetch fines within the date range for this employee
        $fines = Fine::where('employee_id', $employeeId)
            ->whereBetween('date', [$validated['from'], $validated['to']])
            ->get();

        // Return the fines data as JSON
        return response()->json([
            'status' => 'success',
            'data'   => $fines
        ]);
    }

    public function salary_destroy($id)
    {
        $salary = Salary::findOrFail($id);

        $salary->delete();

        return redirect()->route('salary.index')->with('success', 'Fine deleted successfully.');
    }

    public function getPayrollReceipt($id)
    {
        $payroll = Payroll::with('employee')->findOrFail($id);

        $fines = Fine::where('employee_id', $payroll->employee_id)
            ->whereYear('date', $payroll->pay_year)
            ->whereMonth('date', $payroll->pay_month)
            ->get();

        // Calculate approved leave days for this payroll period
        $year = $payroll->pay_year;
        $month = $payroll->pay_month;
        $employeeId = $payroll->employee_id;

        // Get all approved leaves that overlap with this month
        $startOfMonth = \Carbon\Carbon::create($year, $month, 1)->startOfMonth();
        $endOfMonth = \Carbon\Carbon::create($year, $month, 1)->endOfMonth();

        $approvedLeaves = LeaveRequest::where('employee_id', $employeeId)
            ->whereIn('status', [1, 4]) // Include both approved (1) and approved as unpaid (4)
            ->where(function ($query) use ($startOfMonth, $endOfMonth) {
                $query->whereBetween('from', [$startOfMonth, $endOfMonth])
                      ->orWhereBetween('to', [$startOfMonth, $endOfMonth])
                      ->orWhere(function ($q) use ($startOfMonth, $endOfMonth) {
                          $q->where('from', '<', $startOfMonth)
                            ->where('to', '>', $endOfMonth);
                      });
            })
            ->get();

        $approvedLeave = 0;
        $approvedLeaveHours = 0;
        foreach ($approvedLeaves as $leave) {
            $leavePackage = LeavePackage::find($leave->leave_package_id);
            if ($leavePackage) {
                $leaveType = LeaveType::find($leavePackage->leave_type_id);
                if ($leaveType && $leaveType->leave_hours > 0) {
                    $from = \Carbon\Carbon::parse($leave->from)->greaterThan($startOfMonth) ? \Carbon\Carbon::parse($leave->from) : $startOfMonth;
                    $to = \Carbon\Carbon::parse($leave->to)->lessThan($endOfMonth) ? \Carbon\Carbon::parse($leave->to) : $endOfMonth;
                    $days = $from->diffInDays($to) + 1;
                    $approvedLeave += $days;
                    $approvedLeaveHours += $days * $leaveType->leave_hours;
                }
            }
        }

        $payroll->approvedLeave = $approvedLeave;
        $payroll->approved_leave_hours = $approvedLeaveHours;

        return response()->json(['payroll' => $payroll, 'fines' => $fines]);
    }
    
    public function selectEmployeesForPayslip(Request $request)
    {
        $monthFilter = $request->input('month', now()->subMonth()->format('Y-m'));
        $employeeSearch = $request->input('employee_search', '');
        $date = \Carbon\Carbon::createFromFormat('Y-m', $monthFilter);
        $year = $date->year;
        $month = $date->month;

        // Eager load EmpPersonalDetails for filtering
        $employeesQuery = EmpCompanyDetails::with('EmpPersonalDetails')
            ->where('del', 0)
            ->where('compeleted', 1)
            ->where('status', 1);

        // Filter by employee name if search is provided
        if (!empty($employeeSearch)) {
            $employeesQuery->whereHas('EmpPersonalDetails', function ($q) use ($employeeSearch) {
                $q->where('first_name', 'like', '%' . $employeeSearch . '%')
                  ->orWhere('middle_name', 'like', '%' . $employeeSearch . '%')
                  ->orWhere('last_name', 'like', '%' . $employeeSearch . '%');
            });
        }

        $employees = $employeesQuery->get();

        $payrolls = Payroll::where('pay_year', $year)
            ->where('pay_month', $month)
            ->get()
            ->keyBy('employee_id');

        $empList = $employees->map(function ($employee) use ($payrolls, $year, $month) {
            $employee->payslip = $payrolls->get($employee->id);

            // Find salary for the selected month
            $salary = Salary::where('employee_id', $employee->id)
                ->where('from', '<=', sprintf('%04d-%02d', $year, $month))
                ->where('to', '>=', sprintf('%04d-%02d', $year, $month))
                ->first();

            $employee->salary = $salary;
            return $employee;
        });

        return view('Payroll.generate_slip_view', [
            'empList' => $empList,
            'monthFilter' => $monthFilter
        ]);
    }

    public function generateSelectedSlips(Request $request)
    {
        $request->validate([
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'exists:emp_company_details,id',
            'month' => 'required|date_format:Y-m',
        ]);
    
        $employeeIds = $request->employee_ids;
        $month = $request->month;
        $date = Carbon::createFromFormat('Y-m', $month);
    
        try {
            $startDate = $date->startOfMonth()->toDateString();
            $endDate   = $date->copy()->endOfMonth()->toDateString();
        
            $employees = EmpCompanyDetails::whereIn('id', $employeeIds)->get();
        
            foreach ($employees as $employee) {
                $attendances = EmployeeAttendance::where('employee_id', $employee->id)
                    ->whereBetween('date', [$startDate, $endDate])
                    ->get();
        
                $overtimeRecords = Overtime::where('employee_id', $employee->id)
                    ->where('status', 2)
                    ->whereBetween('date', [$startDate, $endDate])
                    ->get();

                $actulovertime = 0;
                $totalMinutes = 0;
                $actualMintus = 0;
                
                // --- LATE FINE CONFIG ---
                $lateFineStart = config('constants.late_fine_start_time', '08:15');
                $lateFineIncrement = (int) config('constants.late_fine_increment', 15);
                $lateFinePercent = (int) config('constants.late_fine_percent', 10);
                $maxIncrements = (int) config('constants.late_fine_max_increments', 0); // 0 means no limit

                // Get salary for daily wage calculation
                $salary = Salary::where('employee_id', $employee->id)
                    ->where(function ($query) use ($startDate, $endDate) {
                        $query->where('from', '<=', \Carbon\Carbon::parse($endDate)->format('Y-m'))
                              ->where('to', '>=', \Carbon\Carbon::parse($startDate)->format('Y-m'));
                    })
                    ->first();

                $basicSalary = $salary ? $salary->basic_salary : 0;
                $workingMinutes = $salary ? $this->calculateScheduledWorkingHours($employee->id, $startDate, $endDate) : 0;

                $workingDays = RosterAssign::where('assign_to', $employee->id)
                    ->whereBetween('schedule_date', [$startDate, $endDate])
                    ->count();
                $dailyWage = ($workingDays > 0) ? ($basicSalary / $workingDays) : 0;

                    foreach ($attendances as $attendance) {
                    if ($attendance->check_in && $attendance->check_out) {
                        $minutes = $attendance->working_hours;
                        $actualMintus += $minutes;
                        $totalMinutes += $minutes;

                        $attendanceController = app(AttendanceController::class);
                        $attendanceController->applyLateFineOnCheckIn($attendance, roster: null); // or pass $roster if available
                    }
            }

                $approvedLeaves = LeaveRequest::where('employee_id', $employee->id)
                ->whereIn('status', [1, 4]) // Include both approved (1) and approved as unpaid (4)
                ->where(function ($query) use ($startDate, $endDate) {
                    $query->whereBetween('from', [$startDate, $endDate])
                        ->orWhereBetween('to', [$startDate, $endDate])
                        ->orWhere(function ($q) use ($startDate, $endDate) {
                            $q->where('from', '<', $startDate)
                                ->where('to', '>', $endDate);
                        });
                })
                ->get();
                foreach ($approvedLeaves as $leave) {
                    $leavePkg = LeavePackage::find($leave->leave_package_id);
                    if ($leavePkg && $leavePkg->leave_type_id) {
                        $leaveType = LeaveType::find($leavePkg->leave_type_id);
                        if ($leaveType && $leaveType->leave_hours) {
                            // Calculate overlap days between leave and period
                            $leaveStart = Carbon::parse(max($leave->from, $startDate));
                            $leaveEnd = Carbon::parse(min($leave->to, $endDate));
                            $days = $leaveStart->diffInDays($leaveEnd) + 1;

                            // If custom-approved with specific paid dates, count only paid dates within this period
                            $paidDaysInPeriod = $days;
                            if (!empty($leave->paid_dates)) {
                                $paidDates = json_decode($leave->paid_dates, true) ?: [];
                                $paidDaysInPeriod = 0;
                                foreach ($paidDates as $d) {
                                    if ($d >= $leaveStart->toDateString() && $d <= $leaveEnd->toDateString()) {
                                        $paidDaysInPeriod++;
                                    }
                                }
                            }

                            $totalMinutes += $paidDaysInPeriod * $leaveType->leave_hours * 60;
                        }
                    }
                }
                $holidays = PublicHoliday::where(function ($query) use ($startDate, $endDate) {
                    $query->whereBetween('from', [$startDate, $endDate])
                        ->orWhereBetween('to', [$startDate, $endDate])
                        ->orWhere(function ($q) use ($startDate, $endDate) {
                            $q->where('from', '<', $startDate)
                                ->where('to', '>', $endDate);
                        });
                })->get();

                $holidayDates = [];
                foreach ($holidays as $holiday) {
                    $period = CarbonPeriod::create($holiday->from, $holiday->to);
                    foreach ($period as $date) {
                        $holidayDates[] = $date->format('Y-m-d');
                    }
                }
                $holidayDates = array_unique($holidayDates);

                // Get employee's scheduled working days from roster
                $scheduledWorkingDays = RosterAssign::where('assign_to', $employee->id)
                    ->whereBetween('schedule_date', [$startDate, $endDate])
                    ->pluck('schedule_date')
                    ->toArray();

                // Get employee's employment start date
                $employmentStartDate = $employee->employment_start_date ? Carbon::parse($employee->employment_start_date) : null;

                // For each holiday, check if it falls on a scheduled working day and employee was employed
                $attendances = EmployeeAttendance::where('employee_id', $employee->id)
                    ->whereBetween('date', [$startDate, $endDate])
                    ->orderBy('date', 'desc')
                    ->get();
                
                foreach ($holidayDates as $holidayDate) {
                    // Check if holiday falls on a scheduled working day
                    $isScheduledWorkingDay = in_array($holidayDate, $scheduledWorkingDays);
                    
                    // Check if employee was employed on this holiday date
                    $wasEmployed = !$employmentStartDate || Carbon::parse($holidayDate)->gte($employmentStartDate);
                    
                    // Check if employee was present on this holiday
                    $wasPresent = $attendances->where('date', $holidayDate)
                        ->whereNotNull('check_in')
                        ->whereNotNull('check_out')
                        ->count() > 0;
                    
                    // Add 480 minutes only if:
                    // 1. Holiday falls on a scheduled working day
                    // 2. Employee was employed on that date
                    // 3. Employee was not present (didn't work on holiday)
                    // if ($isScheduledWorkingDay && $wasEmployed && !$wasPresent) {
                    //     $totalMinutes += 480; // 8 hours
                    // }
                }
                foreach ($overtimeRecords as $overtimeEntry) {
                    if ($overtimeEntry->working_hours) {
                        $overtimeMinutes = $overtimeEntry->working_hours;
                        $actulovertime += $overtimeMinutes;
                        $totalMinutes += $overtimeMinutes;
                    }
                }
        
                $hours            = intdiv($totalMinutes, 60);
                $remainingMinutes = $totalMinutes % 60;
        
                $salaries = Salary::where('employee_id', $employee->id)
                    ->where(function ($query) use ($startDate, $endDate) {
                        $query->where('from', '<=', \Carbon\Carbon::parse($endDate)->format('Y-m'))
                            ->where('to', '>=', \Carbon\Carbon::parse($startDate)->format('Y-m'));
                    })
                    ->get();
                foreach ($salaries as $salary) {
                    // Calculate effective working minutes based on employment start date
                    $effectiveWorkingMinutes = $this->calculateEffectiveWorkingMinutes($employee->id, $startDate, $endDate);
                    // Calculate per-minute rate based on effective working days
                    // $perMinuteRate = $effectiveWorkingMinutes > 0 ? $salary->basic_salary / $effectiveWorkingMinutes : 0;
                    // $calculatedSalary = round($perMinuteRate * $totalMinutes);
                    // Ensure calculated salary doesn't exceed the basic salary for the effective period
                    $effectiveWorkingDays = $this->calculateEffectiveWorkingDays($employee->id, $startDate, $endDate);
                    $totalWorkingDays = $this->calculateWorkingDays($startDate, $endDate);
                    $new_per_hour_rate = ($salary->basic_salary / $totalWorkingDays)/60;
                    $proRatedBasicSalary = $new_per_hour_rate > 0 ? ($new_per_hour_rate * $totalMinutes):0;
                    // Cap the calculated salary to the pro-rated basic salary
                 
                        $calculatedSalary = $proRatedBasicSalary;
                    
                    
                    $existingPayroll = Payroll::where('employee_id', $employee->id)
                        ->where('pay_year', $date->year)
                        ->where('pay_month', $date->month)
                        ->first();
        
                    if (!$existingPayroll) {
                        Payroll::create([
                            'employee_id'           => $employee->id,
                            'pay_year'              => $date->year,
                            'pay_month'             => $date->month,
                            'basic_salary'          => $salary->basic_salary,
                            'working_hours'         => $effectiveWorkingMinutes,
                            'hours_spent'           => $totalMinutes,
                            'actual_working_hours'  => $actualMintus,
                            'overtime_hours'        => $actulovertime,
                            'calculated_salary'     => $calculatedSalary,
                            'status'                => 0
                        ]);
                    }
                }
            }
        
            return response()->json(['success' => true, 'message' => 'Salaries generated successfully.']);

        } catch (\Exception $e) {
            Log::error('Payslip generation failed: ' . $e->getMessage());
            if ($request->ajax()) {
                return response()->json(['success' => false, 'message' => 'An error occurred during payslip generation.'], 500);
            }
            return redirect()->back()->with('error', 'An error occurred during payslip generation.');
        }
    }
        public function bulkDelete(Request $request)
    {
        $request->validate([
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'exists:emp_company_details,id',
            'month' => 'required|date_format:Y-m',
        ]);

        $employeeIds = $request->employee_ids;
        $month = $request->month;
        $date = \Carbon\Carbon::createFromFormat('Y-m', $month);

        $year = $date->year;
        $monthNum = $date->month;

        $deletedCount = 0;

        foreach ($employeeIds as $employeeId) {
            $payroll = Payroll::where('employee_id', $employeeId)
                ->where('pay_year', $year)
                ->where('pay_month', $monthNum)
                ->first();

            if ($payroll) {
                // Delete related fines for this payroll period
                Fine::where('employee_id', $employeeId)
                    ->whereYear('date', $year)
                    ->whereMonth('date', $monthNum)
                    ->delete();

                $payroll->delete();
                $deletedCount++;
            }
        }

        return response()->json([
            'success' => true,
            'message' => "$deletedCount payslip(s) deleted successfully."
        ]);
    }
    public function calculateScheduledWorkingHours($employeeId, $startDate, $endDate)
    {
        $totalMinutes = 0;
        $period = new \DatePeriod(
            new \DateTime($startDate),
            new \DateInterval('P1D'),
            (new \DateTime($endDate))->modify('+1 day')
        );

        foreach ($period as $date) {
            $dateStr = $date->format('Y-m-d');
            $rosterAssign = RosterAssign::where('assign_to', $employeeId)
                ->where('schedule_date', $dateStr)
                ->first();

            if ($rosterAssign) {
                $roster = RosterTemplate::find($rosterAssign->roster_template_id);
                if ($roster) {
                    $minutes = $roster->working_hours;
                    $totalMinutes += max(0, $minutes);
                }
            }
        }
        return $totalMinutes;
    }

    /**
     * Calculate working days excluding weekends and holidays for a given period
     */
    public function calculateWorkingDays($startDate, $endDate, $employeeId = null)
    {
        $workingDays = 0;
        $period = new \DatePeriod(
            new \DateTime($startDate),
            new \DateInterval('P1D'),
            (new \DateTime($endDate))->modify('+1 day')
        );

        // Get holidays for the period
        $holidays = PublicHoliday::where(function ($query) use ($startDate, $endDate) {
            $query->whereBetween('from', [$startDate, $endDate])
                ->orWhereBetween('to', [$startDate, $endDate])
                ->orWhere(function ($q) use ($startDate, $endDate) {
                    $q->where('from', '<', $startDate)
                        ->where('to', '>', $endDate);
                });
        })->get();

        $holidayDates = [];
        foreach ($holidays as $holiday) {
            $holidayPeriod = CarbonPeriod::create($holiday->from, $holiday->to);
            foreach ($holidayPeriod as $date) {
                $holidayDates[] = $date->format('Y-m-d');
            }
        }
        $holidayDates = array_unique($holidayDates);

        foreach ($period as $date) {
            $dateStr = $date->format('Y-m-d');
            $dayOfWeek = $date->format('N'); // 1 (Monday) through 7 (Sunday)
            
            // Skip weekends (Saturday = 6, Sunday = 7)
            if ($dayOfWeek >= 6) {
                continue;
            }
            
            // Skip holidays
            if (in_array($dateStr, $holidayDates)) {
                continue;
            }
            
            $workingDays++;
        }
        return $workingDays *8;
    }

    /**
     * Calculate effective working days for an employee based on employment start date
     */
    public function calculateEffectiveWorkingDays($employeeId, $startDate, $endDate)
    {
        // Get employee's employment start date
        $employee = EmpCompanyDetails::find($employeeId);
        if (!$employee || !$employee->employment_start_date) {
            // If no employment start date, use the full period
            return $this->calculateWorkingDays($startDate, $endDate, $employeeId);
        }

        $employmentStartDate = Carbon::parse($employee->employment_start_date);
        $periodStartDate = Carbon::parse($startDate);
        $periodEndDate = Carbon::parse($endDate);

        // If employee started before the period, use full period
        if ($employmentStartDate->lte($periodStartDate)) {
            return $this->calculateWorkingDays($startDate, $endDate, $employeeId);
        }

        // If employee started after the period, return 0
        if ($employmentStartDate->gt($periodEndDate)) {
            return 0;
        }

        // Employee started during the period, calculate from employment start date
        $effectiveStartDate = $employmentStartDate->format('Y-m-d');
        return $this->calculateWorkingDays($effectiveStartDate, $endDate, $employeeId);
    }

    /**
     * Calculate effective working minutes for salary calculation
     */
    public function calculateEffectiveWorkingMinutes($employeeId, $startDate, $endDate)
    {
        $effectiveWorkingDays = $this->calculateEffectiveWorkingDays($employeeId, $startDate, $endDate);
        // Assuming 8 hours per working day (480 minutes)
        return $effectiveWorkingDays * 60;
    }

    /**
     * Test method to verify salary calculation logic
     */
    public function testSalaryCalculation($employeeId, $startDate, $endDate)
    {
        $employee = EmpCompanyDetails::find($employeeId);
        $effectiveWorkingDays = $this->calculateEffectiveWorkingDays($employeeId, $startDate, $endDate);
        $totalWorkingDays = $this->calculateWorkingDays($startDate, $endDate);
        $effectiveWorkingMinutes = $this->calculateEffectiveWorkingMinutes($employeeId, $startDate, $endDate);
        
        return [
            'employee_id' => $employeeId,
            'employee_name' => $employee ? ($employee->empPersonalDetails->first_name ?? 'Unknown') : 'Unknown',
            'employment_start_date' => $employee ? $employee->employment_start_date : 'Not set',
            'period_start' => $startDate,
            'period_end' => $endDate,
            'total_working_days' => $totalWorkingDays,
            'effective_working_days' => $effectiveWorkingDays,
            'effective_working_minutes' => $effectiveWorkingMinutes,
            'pro_rated_percentage' => $totalWorkingDays > 0 ? round(($effectiveWorkingDays / $totalWorkingDays) * 100, 2) : 0
        ];
    }

    public function getRosterWorkingHours(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|integer',
            'year' => 'required|integer',
            'month' => 'required|integer|between:1,12',
        ]);

        $employeeId = $request->employee_id;
        $year = $request->year;
        $month = $request->month;

        // Create date range for the month
        $startDate = Carbon::createFromDate($year, $month, 1)->startOfMonth()->toDateString();
        $endDate = Carbon::createFromDate($year, $month, 1)->endOfMonth()->toDateString();

        // Calculate scheduled working hours using the same method as in generateSelectedSlips
        $workingMinutes = $this->calculateScheduledWorkingHours($employeeId, $startDate, $endDate);

        return response()->json([
            'working_hours' => $workingMinutes,
            'working_hours_formatted' => round($workingMinutes / 60, 2)
        ]);
    }
    public function viewDetails($id)
    {
        $payroll = Payroll::with(['employee', 'employeCom'])->findOrFail($id);

        // Dates for the payroll period
        $startDate = Carbon::createFromDate($payroll->pay_year, $payroll->pay_month, 1)->startOfMonth();
        $endDate = Carbon::createFromDate($payroll->pay_year, $payroll->pay_month, 1)->endOfMonth();

        // Fines and bonuses
        $fines = Fine::where('employee_id', $payroll->employee_id)
            ->whereYear('date', $payroll->pay_year)
            ->whereMonth('date', $payroll->pay_month)
            ->orderBy('date', 'desc')
            ->get();

        // Overtime
        $overtimeRecords = Overtime::where('employee_id', $payroll->employee_id)
            ->where('status', 2)
            ->whereBetween('date', [$startDate->toDateString(), $endDate->toDateString()])
            ->orderBy('date', 'desc')
            ->get();

        // Salary
        $salary = Salary::where('employee_id', $payroll->employee_id)
            ->where(function ($query) use ($startDate, $endDate) {
                $query->where('from', '<=', $endDate->format('Y-m'))
                    ->where('to', '>=', $startDate->format('Y-m'));
            })
            ->first();
        // Totals
        $totalFines = $fines->where('type', 0)->sum('fine_amount');
        $totalBonuses = $fines->where('type', 1)->sum('fine_amount');
        $adjustedSalary = $payroll->calculated_salary - $totalFines + $totalBonuses;
        // Holidays (get all dates in the period)
        $holidays = PublicHoliday::where(function ($query) use ($startDate, $endDate) {
            $query->whereBetween('from', [$startDate, $endDate])
                ->orWhereBetween('to', [$startDate, $endDate])
                ->orWhere(function ($q) use ($startDate, $endDate) {
                    $q->where('from', '<', $startDate)
                        ->where('to', '>', $endDate);
                });
        })->get();

        $holidayDates = [];
        foreach ($holidays as $holiday) {
            $period = CarbonPeriod::create($holiday->from, $holiday->to);
            foreach ($period as $date) {
                $holidayDates[] = $date->format('Y-m-d');
            }
        }
        $holidayDates = array_unique($holidayDates);

        // Scheduled working days (from roster)
        $scheduledDays = RosterAssign::where('assign_to', $payroll->employee_id)
            ->whereBetween('schedule_date', [$startDate->toDateString(), $endDate->toDateString()])
            ->pluck('schedule_date')
            ->toArray();
        $scheduledDays = array_values(array_unique($scheduledDays));
        // Consider only holidays that fall on scheduled working days
        $holidayOnScheduledDays = array_values(array_intersect($scheduledDays, $holidayDates));
        // Remove holidays from scheduled working days
        $actualWorkingDays = array_diff($scheduledDays, $holidayOnScheduledDays);
        $totalWorkingDays = count($actualWorkingDays);
        $totalHolidays = count($holidayOnScheduledDays);
        // Attendance: present days (on actual working days)
        $presentDays = EmployeeAttendance::where('employee_id', $payroll->employee_id)
            ->whereIn('date', $actualWorkingDays)
            ->whereNotNull('check_in')
            ->whereNotNull('check_out')
            ->distinct('date')
            ->count('date');
        // Absent days
        $absentDays = $totalWorkingDays - $presentDays;
        // Count present days plus holidays that are on scheduled days only
        $actualPresentDays = $presentDays + $totalHolidays;
        // Attendance rate
        $attendanceRate = $totalWorkingDays > 0 ? round(($presentDays / $totalWorkingDays) * 100, 1) : 0;

        // Attendance records (for details table, if needed)
        $attendances = EmployeeAttendance::where('employee_id', $payroll->employee_id)
            ->whereBetween('date', [$startDate->toDateString(), $endDate->toDateString()])
            ->orderBy('date', 'desc')
            ->get();

        // Calculate approved leave days and hours for this payroll period
        $approvedLeaves = LeaveRequest::where('employee_id', $payroll->employee_id)
            ->whereIn('status', [1, 4]) // Include both approved (1) and approved as unpaid (4)
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('from', [$startDate, $endDate])
                      ->orWhereBetween('to', [$startDate, $endDate])
                      ->orWhere(function ($q) use ($startDate, $endDate) {
                          $q->where('from', '<', $startDate)
                            ->where('to', '>', $endDate);
                      });
            })
            ->get();

        $approvedLeave = 0;
        $approvedLeaveHours = 0;
        $paidLeaveHours = 0;
        $unpaidLeaveHours = 0;

        foreach ($approvedLeaves as $leave) {
            $leavePackage = LeavePackage::find($leave->leave_package_id);
            if ($leavePackage) {
                $leaveType = LeaveType::find($leavePackage->leave_type_id);
                if ($leaveType && $leaveType->leave_hours > 0) {
                    // Calculate overlap days between leave and period
                    $from = \Carbon\Carbon::parse($leave->from)->greaterThan($startDate) ? \Carbon\Carbon::parse($leave->from) : $startDate;
                    $to = \Carbon\Carbon::parse($leave->to)->lessThan($endDate) ? \Carbon\Carbon::parse($leave->to) : $endDate;
                    $days = $from->diffInDays($to) + 1;

                    $paidDaysInPeriod = $days;
                    $unpaidDaysInPeriod = 0;
                    
                    // Handle custom paid dates or status-based leave calculation
                    if (!empty($leave->paid_dates)) {
                        $paidDaysInPeriod = 0;
                        $paidDates = json_decode($leave->paid_dates, true) ?: [];
                        foreach ($paidDates as $d) {
                            if ($d >= $from->toDateString() && $d <= $to->toDateString()) {
                                $paidDaysInPeriod++;
                            }
                        }
                        $unpaidDaysInPeriod = $days - $paidDaysInPeriod;
                    } elseif ($leave->status == 4) { // Approved as unpaid
                        $paidDaysInPeriod = 0;
                        $unpaidDaysInPeriod = $days;
                    }
                    
                    $approvedLeave += $paidDaysInPeriod;
                    $approvedLeaveHours += $paidDaysInPeriod * $leaveType->leave_hours;
                    $paidLeaveHours += $paidDaysInPeriod * $leaveType->leave_hours;
                    $unpaidLeaveHours += $unpaidDaysInPeriod * $leaveType->leave_hours;
                }
            }
        }

        // Calculate total working hours (scheduled working hours for the period)
        $totalWorkingHours = $this->calculateScheduledWorkingHours($payroll->employee_id, $startDate->toDateString(), $endDate->toDateString());
        
        // Calculate paid worked hours (actual working hours + paid leave hours)
        $paidWorkedHours = ($payroll->actual_working_hours / 60) + $paidLeaveHours;

        return view('Payroll.details', compact(
            'payroll',
            'fines',
            'attendances',
            'overtimeRecords',
            'salary',
            'totalFines',
            'totalBonuses',
            'adjustedSalary',
            'totalWorkingDays',
            'totalHolidays',
            'presentDays',
            'absentDays',
            'attendanceRate',
            'holidays',
            'approvedLeave',
            'approvedLeaveHours',
            'paidLeaveHours',
            'unpaidLeaveHours',
            'totalWorkingHours',
            'paidWorkedHours'
        ));
    }

    public function savePayrollPdf(Request $request, $id)
    {
        try {
            $payroll = Payroll::with('employee')->findOrFail($id);

            // Accept the uploaded file (no strict mime check, since browser may not set it)
            $request->validate([
                'pdf' => 'required',
            ]);

            // Get month and year folder name
            $monthName = date('F', mktime(0, 0, 0, $payroll->pay_month, 10)); // e.g., January
            $folderName = $monthName . '_' . $payroll->pay_year; // e.g., January_2025
            $storagePath = 'public/payroll_receipts/' . $folderName;

            // Get employee name for filename
            $emp = $payroll->employee ?? null;
            $empName = '';
            if ($emp) {
                $empName = trim(($emp->first_name ?? '') . '_' . ($emp->middle_name ?? '') . '_' . ($emp->last_name ?? ''));
                $empName = preg_replace('/[^A-Za-z0-9_]/', '', str_replace(' ', '_', $empName));
                $empName = preg_replace('/_+/', '_', $empName); // Remove duplicate underscores
            } else {
                $empName = 'Employee_' . $payroll->employee_id;
            }

            if ($request->hasFile('pdf')) {
                $file = $request->file('pdf');
                $filename = "{$empName}_{$payroll->pay_year}_{$payroll->pay_month}.pdf";
                $path = $file->storeAs($storagePath, $filename);

                // Update payroll record with PDF path (remove 'public/' for web access)
                $payroll->update([
                    'pdf_path' => 'payroll_receipts/' . $folderName . '/' . $filename
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'PDF saved successfully',
                    'pdf_path' => 'payroll_receipts/' . $folderName . '/' . $filename
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'No PDF file uploaded'
                ], 400);
            }
        } catch (\Exception $e) {
            Log::error('Error saving payroll PDF: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    public function downloadPayrollPdf($id)
    {
        try {
            $payroll = Payroll::findOrFail($id);
            
            if (!$payroll->pdf_path) {
                return response()->json([
                    'success' => false,
                    'message' => 'PDF not found'
                ], 404);
            }

            $filepath = storage_path('app/public/' . $payroll->pdf_path);
            
            if (!file_exists($filepath)) {
                return response()->json([
                    'success' => false,
                    'message' => 'PDF file not found on server'
                ], 404);
            }

            return response()->download($filepath);

        } catch (\Exception $e) {
            Log::error('Error downloading payroll PDF: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error downloading PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    public function attendancePdf($id)
    {
        $payroll = Payroll::with(['employee', 'employeCom'])->findOrFail($id);

        // Dates for the payroll period
        $startDate = Carbon::createFromDate($payroll->pay_year, $payroll->pay_month, 1)->startOfMonth();
        $endDate = Carbon::createFromDate($payroll->pay_year, $payroll->pay_month, 1)->endOfMonth();

        // Fines and bonuses
        $fines = Fine::where('employee_id', $payroll->employee_id)
            ->whereYear('date', $payroll->pay_year)
            ->whereMonth('date', $payroll->pay_month)
            ->orderBy('date', 'desc')
            ->get();

        // Overtime
        $overtimeRecords = Overtime::where('employee_id', $payroll->employee_id)
            ->where('status', 2)
            ->whereBetween('date', [$startDate->toDateString(), $endDate->toDateString()])
            ->orderBy('date', 'desc')
            ->get();

        // Salary
        $salary = Salary::where('employee_id', $payroll->employee_id)
            ->where(function ($query) use ($startDate, $endDate) {
                $query->where('from', '<=', $endDate->format('Y-m'))
                    ->where('to', '>=', $startDate->format('Y-m'));
            })
            ->first();

        // Totals
        $totalFines = $fines->where('type', 0)->sum('fine_amount');
        $totalBonuses = $fines->where('type', 1)->sum('fine_amount');
        $adjustedSalary = $payroll->calculated_salary - $totalFines + $totalBonuses;

        // Holidays (get all dates in the period)
        $holidays = PublicHoliday::where(function ($query) use ($startDate, $endDate) {
            $query->whereBetween('from', [$startDate, $endDate])
                ->orWhereBetween('to', [$startDate, $endDate])
                ->orWhere(function ($q) use ($startDate, $endDate) {
                    $q->where('from', '<', $startDate)
                        ->where('to', '>', $endDate);
                });
        })->get();

        $holidayDates = [];
        foreach ($holidays as $holiday) {
            $period = CarbonPeriod::create($holiday->from, $holiday->to);
            foreach ($period as $date) {
                $holidayDates[] = $date->format('Y-m-d');
            }
        }
        $holidayDates = array_unique($holidayDates);

        // Scheduled working days (from roster)
        $scheduledDays = RosterAssign::where('assign_to', $payroll->employee_id)
            ->whereBetween('schedule_date', [$startDate->toDateString(), $endDate->toDateString()])
            ->pluck('schedule_date')
            ->toArray();

        // Remove holidays from scheduled working days
        $actualWorkingDays = array_diff($scheduledDays, $holidayDates);
        $totalWorkingDays = count($actualWorkingDays);
        $totalHolidays = count($holidayDates);

        // Attendance: present days (on actual working days)
        $presentDays = EmployeeAttendance::where('employee_id', $payroll->employee_id)
            ->whereIn('date', $actualWorkingDays)
            ->whereNotNull('check_in')
            ->whereNotNull('check_out')
            ->distinct('date')
            ->count('date');

        // Absent days
        $absentDays = $totalWorkingDays - $presentDays;

        // Attendance rate
        $attendanceRate = $totalWorkingDays > 0 ? round(($presentDays / $totalWorkingDays) * 100, 1) : 0;

        // Attendance records (for details table, if needed)
        $attendances = EmployeeAttendance::where('employee_id', $payroll->employee_id)
            ->whereBetween('date', [$startDate->toDateString(), $endDate->toDateString()])
            ->orderBy('date', 'desc')
            ->get();

        // Calculate approved leave days and hours for this payroll period
        $approvedLeaves = LeaveRequest::where('employee_id', $payroll->employee_id)
            ->whereIn('status', [1, 4]) // Include both approved (1) and approved as unpaid (4)
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('from', [$startDate, $endDate])
                      ->orWhereBetween('to', [$startDate, $endDate])
                      ->orWhere(function ($q) use ($startDate, $endDate) {
                          $q->where('from', '<', $startDate)
                            ->where('to', '>', $endDate);
                      });
            })
            ->get();

        $approvedLeave = 0;
        $approvedLeaveHours = 0;

        foreach ($approvedLeaves as $leave) {
            $leavePackage = LeavePackage::find($leave->leave_package_id);
            if ($leavePackage) {
                $leaveType = LeaveType::find($leavePackage->leave_type_id);
                if ($leaveType && $leaveType->leave_hours > 0) {
                    // Calculate overlap paid days between leave and period
                    $from = \Carbon\Carbon::parse($leave->from)->greaterThan($startDate) ? \Carbon\Carbon::parse($leave->from) : $startDate;
                    $to = \Carbon\Carbon::parse($leave->to)->lessThan($endDate) ? \Carbon\Carbon::parse($leave->to) : $endDate;
                    $days = $from->diffInDays($to) + 1;

                    $paidDaysInPeriod = $days;
                    if (!empty($leave->paid_dates)) {
                        $paidDaysInPeriod = 0;
                        $paidDates = json_decode($leave->paid_dates, true) ?: [];
                        foreach ($paidDates as $d) {
                            if ($d >= $from->toDateString() && $d <= $to->toDateString()) {
                                $paidDaysInPeriod++;
                            }
                        }
                    }

                    $approvedLeave += $paidDaysInPeriod;
                    $approvedLeaveHours += $paidDaysInPeriod * $leaveType->leave_hours;
                }
            }
        }
        
        $html = view('Payroll.attendance_pdf', compact(
            'payroll',
            'fines',
            'attendances',
            'overtimeRecords',
            'salary',
            'totalFines',
            'totalBonuses',
            'adjustedSalary',
            'totalWorkingDays',
            'totalHolidays',
            'presentDays',
            'absentDays',
            'attendanceRate',
            'holidays',
            'approvedLeave',
            'approvedLeaveHours'
        ))->render();

        $pdf = \PDF::loadHTML($html)->setPaper('a4', 'portrait');

        // Save to storage
        $monthName = date('F', mktime(0, 0, 0, $payroll->pay_month, 10));
        $folderName = $monthName . '_' . $payroll->pay_year;
        $storagePath = 'public/attendance_pdfs/' . $folderName;
        $emp = $payroll->employee ?? null;
        $empName = $emp ? trim(($emp->first_name ?? '') . '_' . ($emp->middle_name ?? '') . '_' . ($emp->last_name ?? '')) : 'Employee_' . $payroll->employee_id;
        $empName = preg_replace('/[^A-Za-z0-9_]/', '', str_replace(' ', '_', $empName));
        $empName = preg_replace('/_+/', '_', $empName);
        $filename = "{$empName}_attendance_{$payroll->pay_year}_{$payroll->pay_month}.pdf";
        $fullPath = storage_path("app/{$storagePath}/{$filename}");

        // Ensure directory exists
        if (!file_exists(storage_path("app/{$storagePath}"))) {
            mkdir(storage_path("app/{$storagePath}"), 0755, true);
        }

        $pdf->save($fullPath);

        // Save path in DB (remove 'public/' for web access)
        $payroll->attendance_pdf_path = "attendance_pdfs/{$folderName}/{$filename}";
        $payroll->save();

        return response()->download($fullPath);
    }

    public function sendReceiptEmail(Request $request, $id)
    {
        $payroll = Payroll::with(['employee', 'employeCom'])->find($id);
        if (!$payroll || !$payroll->employee) {
            return response()->json(['success' => false, 'message' => 'Payroll or employee not found.'], 404);
        }
        $email = $payroll->employeCom->employee_email ?? $payroll->employeCom->email ?? null;
        if (!$email) {
            return response()->json(['success' => false, 'message' => 'No email address found for this employee.'], 400);
        }
        if (!$payroll->pdf_path) {
            return response()->json(['success' => false, 'message' => 'Payslip PDF not found.Please generate it!'], 404);
        }
        $pdfPath = storage_path('app/public/' . $payroll->pdf_path);
        if (!file_exists($pdfPath)) {
            return response()->json(['success' => false, 'message' => 'Payslip PDF file not found on server.'], 404);
        }
        
        // Create descriptive filename
        $emp = $payroll->employee ?? null;
        $empName = $emp ? trim(($emp->first_name ?? '') . '_' . ($emp->middle_name ?? '') . '_' . ($emp->last_name ?? '')) : 'Employee_' . $payroll->employee_id;
        $empName = preg_replace('/[^A-Za-z0-9_]/', '', str_replace(' ', '_', $empName));
        $empName = preg_replace('/_+/', '_', $empName);
        $attachmentName = "{$empName}_Payslip_{$payroll->pay_year}_{$payroll->pay_month}.pdf";
        
        $data = [
            'payroll' => $payroll,
            'employee' => $payroll->employee,
            'subject' => 'Your Payslip Receipt | ' . env('APP_NAME'),
        ];

        // Send email with PDF attachment using the new trait method
        $emailSent = $this->sendEmailWithAttachment(
            $email,
            $data['subject'],
            $data,
            'Emails.payslip_receipt',
            $pdfPath,
            $attachmentName,
            'application/pdf'
        );

        if ($emailSent) {
            return response()->json(['success' => true, 'message' => 'Payslip email sent successfully!']);
        } else {
            Log::error('Failed to send payslip email using trait.');
            return response()->json(['success' => false, 'message' => 'Failed to send payslip email.'], 500);
        }
    }
}
 