<?php

namespace App\Jobs;

use App\Http\Controllers\PayRollController;
use App\Models\EmpCompanyDetails;
use App\Models\Fine;
use App\Models\LeavePackage;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Models\Payroll;
use App\Models\Salary;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Http\Request;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Barryvdh\DomPDF\Facade\Pdf as PDF;
use App\Traits\EmailTrait;

class GenerateAllPayrollReceiptsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels, EmailTrait;

    public function handle()
    {
        Log::info('Payroll receipts job started');

        $monthString = Carbon::now()->subMonth()->format('Y-m');
        $date        = Carbon::createFromFormat('Y-m', $monthString);

        $employees = EmpCompanyDetails::where('del', 0)
            ->where('compeleted', 1)
            ->where('status', 1)
            ->get();

        $controller   = app(PayRollController::class);
        $successCount = 0;
        $failures     = [];

        foreach ($employees as $employee) {
            Log::info('Processing employee for payroll receipt', ['employee_id' => $employee->id]);

            $payroll = Payroll::where('employee_id', $employee->id)
                ->where('pay_year',  $date->year)
                ->where('pay_month', $date->month)
                ->first();

            if (!$payroll) {
                $fakeReq = Request::create('/payroll/generate-selected-slips', 'POST', [
                    'employee_ids' => [$employee->id],
                    'month'        => $monthString,
                ]);

                try {
                    $controller->generateSelectedSlips($fakeReq);
                } catch (ValidationException $ve) {
                    Log::warning('Skipped employee due to validation errors', [
                        'employee_id' => $employee->id,
                        'errors'      => $ve->errors(),
                    ]);
                    $failures[] = $employee->id;
                    continue;
                }

                $payroll = Payroll::where('employee_id', $employee->id)
                    ->where('pay_year',  $date->year)
                    ->where('pay_month', $date->month)
                    ->first();
            }

            if (! $payroll) {
                Log::warning('Payroll not generated for employee after controller run', [
                    'employee_id' => $employee->id,
                ]);
                $failures[] = $employee->id;
                continue;
            }

            $this->generateReceiptPdfAndEmail($employee, $payroll);
            $successCount++;
        }
        Log::info('Payroll receipts job completed', [
            'success_count' => $successCount,
            'failures'      => $failures,
        ]);
    }

    protected function generateReceiptPdfAndEmail($employee, $payroll)
    {
        $year      = $payroll->pay_year;
        $month     = $payroll->pay_month;
        $monthName = date('F', mktime(0, 0, 0, $month, 10));
        $folder    = "{$monthName}_{$year}";

        $fines = Fine::where('employee_id', $employee->id)
            ->whereYear('date',  $year)
            ->whereMonth('date', $month)
            ->orderBy('date', 'desc')
            ->get();

        $salary = Salary::where('employee_id', $employee->id)
            ->where(function ($q) use ($year, $month) {
                $period = "{$year}-" . str_pad($month, 2, '0', STR_PAD_LEFT);
                $q->where('from', '<=', $period)
                    ->where('to',   '>=', $period);
            })
            ->first();

        $totalFines   = $fines->where('type', 0)->sum('fine_amount');
        $totalBonuses = $fines->where('type', 1)->sum('fine_amount');
        $adjusted     = $payroll->calculated_salary - $totalFines + $totalBonuses;

        $startOfMonth = Carbon::create($year, $month, 1)->startOfMonth();
        $endOfMonth   = Carbon::create($year, $month, 1)->endOfMonth();

        $approvedLeaves = LeaveRequest::where('employee_id', $employee->id)
            ->whereIn('status', [1, 4]) // Include both approved (1) and approved as unpaid (4)
            ->where(function ($q) use ($startOfMonth, $endOfMonth) {
                $q->whereBetween('from', [$startOfMonth, $endOfMonth])
                    ->orWhereBetween('to',   [$startOfMonth, $endOfMonth])
                    ->orWhere(function ($q2) use ($startOfMonth, $endOfMonth) {
                        $q2->where('from', '<', $startOfMonth)
                            ->where('to', '>', $endOfMonth);
                    });
            })->get();

        $approvedLeaveDays   = 0;
        $approvedLeaveHours  = 0;
        foreach ($approvedLeaves as $leave) {
            $pkg      = LeavePackage::find($leave->leave_package_id);
            $leaveType = $pkg ? LeaveType::find($pkg->leave_type_id) : null;
            if ($leaveType && $leaveType->leave_hours > 0) {
                $from = max($startOfMonth, Carbon::parse($leave->from));
                $to   = min($endOfMonth,   Carbon::parse($leave->to));
                $days = $from->diffInDays($to) + 1;

                $paidDaysInPeriod = $days;
                if (!empty($leave->paid_dates)) {
                    $paidDaysInPeriod = 0;
                    $paidDates = json_decode($leave->paid_dates, true) ?: [];
                    $fromStr = $from->toDateString();
                    $toStr = $to->toDateString();
                    foreach ($paidDates as $d) {
                        if ($d >= $fromStr && $d <= $toStr) {
                            $paidDaysInPeriod++;
                        }
                    }
                }

                $approvedLeaveDays  += $paidDaysInPeriod;
                $approvedLeaveHours += $paidDaysInPeriod * $leaveType->leave_hours;
            }
        }

        $html = view('Payroll.pdf_receipt', [
            'payroll'           => $payroll,
            'employee'          => $employee,
            'salary'            => $salary,
            'fines'             => $fines,
            'totalFines'        => $totalFines,
            'totalBonuses'      => $totalBonuses,
            'adjustedSalary'    => $adjusted,
            'approvedLeave'     => $approvedLeaveDays,
            'approvedLeaveHours' => $approvedLeaveHours,
        ])->render();

        $pdf = PDF::loadHTML($html)->setPaper('a4', 'portrait');

        $path = storage_path("app/public/payroll_receipts/{$folder}");
        if (! is_dir($path)) mkdir($path, 0755, true);

        $empDetails = $employee->EmpPersonalDetails;
        $empName = preg_replace(
            '/[^A-Za-z0-9_]/',
            '',
            implode('_', array_filter([
                $empDetails->first_name  ?? null,
                $empDetails->middle_name ?? null,
                $empDetails->last_name   ?? null,
            ]))
        );
        $filename = "{$empName}_{$year}_{$month}.pdf";
        $fullPath = "{$path}/{$filename}";
        $pdf->save($fullPath);

        $payroll->pdf_path = "payroll_receipts/{$folder}/{$filename}";
        $payroll->save();

        $email = $employee->employee_email ?: $employee->email;
        if ($email) {
            $this->sendEmailWithAttachment(
                $email,
                "Your Payslip | " . config('app.name'),
                compact('employee', 'payroll'),
                'Emails.payslip_receipt',
                $fullPath,
                $filename,
                'application/pdf'
            );
            Log::info('Payslip email sent', [
                'employee_id' => $employee->id,
                'email'       => $email,
            ]);
        }
    }
}
